/*******************************************************************************
* FILE NAME: user_routines_fast.c <EDU VERSION>
*
* DESCRIPTION:
*  This file is where the user can add their custom code within the framework
*  of the routines below. 
*
* USAGE:
*  You can either modify this file to fit your needs, or remove it from your 
*  project and replace it with a modified copy. 
*
* OPTIONS:  Interrupts are disabled and not used by default.
*
*******************************************************************************/

#include "ifi_aliases.h"   // Macros/Aliases used to refer to ports
#include "ifi_default.h"   // Prototypes for default functions
#include "ifi_utilities.h" // Prototypes for utility functions
#include "user_routines.h" // Prototypes for user defined

/*** DEFINE USER VARIABLES AND INITIALIZE THEM HERE ***/
/* Variable "State" remembers state of the robot.  "State" is initialized to 
   I (corresponding with the integer 0), and cycles II-IV (corresponding with integers
   1-3. */
unsigned char State = StateI;
// Variable "infront" remembers which pair of legs is in front
unsigned char infront = OUT;
// Variable "infrontlast" remembers which pair of legs was last in front
unsigned char infrontlast = OUT;
// Variable "inback" remembers which pair of legs is in back
unsigned char inback = IN;

/*******************************************************************************
* FUNCTION NAME: InterruptVectorLow
* PURPOSE:       Low priority interrupt vector
* CALLED FROM:   nowhere by default
* ARGUMENTS:     none
* RETURNS:       void
* DO NOT MODIFY OR DELETE THIS FUNCTION 
*******************************************************************************/
#pragma code InterruptVectorLow = LOW_INT_VECTOR
void InterruptVectorLow (void)
{
  _asm
    goto InterruptHandlerLow  /*jump to interrupt routine*/
  _endasm
}

/*******************************************************************************
* FUNCTION NAME: InterruptHandlerLow
* PURPOSE:       Low priority interrupt handler
* If you want to use these external low priority interrupts or any of the
* peripheral interrupts then you must enable them in your initialization
* routine.  Innovation First, Inc. will not provide support for using these
* interrupts, so be careful.  There is great potential for glitchy code if good
* interrupt programming practices are not followed.  Especially read p. 28 of
* the "MPLAB(R) C18 C Compiler User's Guide" for information on context saving.
* CALLED FROM:   this file, InterruptVectorLow routine
* ARGUMENTS:     none
* RETURNS:       void
*******************************************************************************/
#pragma code

/* You may want to save additional symbols. */
#pragma interruptlow InterruptHandlerLow save=PROD 

void InterruptHandlerLow ()     
{                               
  unsigned char int_byte;       
  if (INTCON3bits.INT2IF)         /* The INT2 pin is RB2/INTERRUPTS 1. */
  { 
    INTCON3bits.INT2IF = 0;
  }
  else if (INTCON3bits.INT3IF)    /* The INT3 pin is RB3/INTERRUPTS 2. */
  {
    INTCON3bits.INT3IF = 0;
  }
  else if (INTCONbits.RBIF)       /* INTERRUPTS 3-4 (RB4, RB5, RB6, or RB7) changed. */
  {
    int_byte = PORTB;             /* You must read or write to PORTB   */
    INTCONbits.RBIF = 0;          /* and clear the interrupt flag      */
  }                               /* to clear the interrupt condition. */
}

/*******************************************************************************
* Processing Functions
*******************************************************************************/

/*******************************************************************************
* FUNCTION NAME: Potentiometer
* PURPOSE:       Read specified potentiometer (Continuous Feedbacks Alpha and Beta)
* CALLED FROM:   Sense Functions
* ARGUMENTS:     motor (IN, OUT, or HIP)
* RETURNS:       potentiometer (reading of specified potentiometer)
*******************************************************************************/

int Potentiometer(char motor)
{
  unsigned int potentiometer;
  switch(motor)
  {
    case IN:
    {
      potentiometer = Get_Analog_Value(PotIn);
      break;
    }
    case OUT:
    {
      potentiometer =  Get_Analog_Value(PotOut);
      break;
    }
    case HIP:
    {  
      potentiometer = Get_Analog_Value(PotHip);
      break;
    }
  }
  return potentiometer;
}

/*******************************************************************************
* FUNCTION NAME: SendPWM
* PURPOSE:       Sets pwm of specified motor to specified value
* CALLED FROM:   Action Functions
* ARGUMENTS:     motor (IN, OUT, or HIP); PWM (signed 8-bit)
* RETURNS:       void
*******************************************************************************/
void SendPWM(char motor, int pwm)
{   
  pwm = 131 + pwm;	// Convert signed 8-bit number to unsigned 8-bit number
  if (pwm >255)		// If value is greater than allowed maximum
  {
   pwm = 255;		// Set to allowed maximum
  }
  else if (pwm<0)	// If value is less than allowed minimum
  {
   pwm = 0;		// Set to allowed minimum
  }
  switch(motor)
  {
    case IN:
    {
      MotorInPWM = pwm;
      break;
    }
    case OUT:
    {
      MotorOutPWM = pwm;
      break;
    }
    case HIP:
    {  
      MotorHipPWM = pwm;
      break;
    }  
  }
}

/*******************************************************************************
* FUNCTION NAME: InFrontInBack
* PURPOSE:       Determine which pair of legs is in front and which is in back
* CALLED FROM:   State Machine Function
* ARGUMENTS:     none
* RETURNS:       void
*******************************************************************************/
void InFrontInBack(void)
{
  if (Potentiometer(HIP) > HIPSWITCH)
  {
    infront = IN;
    inback = OUT;
  }
  if (Potentiometer(HIP) <= HIPSWITCH)
  {
    infront = OUT;
    inback = IN;
  }
}

/*******************************************************************************
* FUNCTION NAME: Position2PWM
* PURPOSE:       For position control, convert current position to recommended 
*                pwm signal
* CALLED FROM:   Sense Functions
* ARGUMENTS:     Ptarget, motor, tolerance, scale, offset
* RETURNS:       pwm
*******************************************************************************/
int Position2PWM(int Ptarget, char motor, char tolerance, char scale, char offset)
{
  int position = Potentiometer(motor);
  int pwm;

  if (position > (Ptarget - tolerance)		// if motor is within Tolerance
    && position < (Ptarget + tolerance))
  { 
    pwm = NEUTRAL;				// turn motor off
  }
// Otherwise, Scale voltage proportional to position error, bit shift, and add offset
  else if (Ptarget-position < 0)			
  {
    pwm = ((int)scale*(Ptarget-position))/80 - (int)offset;
  }
  else
  {
    pwm = ((int)scale*(Ptarget-position))/80 + (int)offset;
  }
  return pwm;
}

/*******************************************************************************
* Sense Functions
*******************************************************************************/

/*******************************************************************************
* FUNCTION NAME: HeelStrike
* PURPOSE:       Determine if heel strike (Sense A) occurs
* CALLED FROM:   State Machine Function
* ARGUMENTS:     none
* RETURNS:       IN, OUT, or NONE
*******************************************************************************/
char HeelStrike (void)
{
  char heelstrike = NONE;
  if (infront == IN & LimitHeelIn == CLOSED)
  {
    heelstrike = IN;
  }
  else if (infront == OUT & LimitHeelOut == CLOSED)
  {
    heelstrike = OUT;
  }
  return heelstrike;
}


/*******************************************************************************
* FUNCTION NAME: ToeLift
* PURPOSE:       Determine if toe lift (Sense B) occurs
* CALLED FROM:   State Machine Function
* ARGUMENTS:     none
* RETURNS:       IN, OUT, or NONE
*******************************************************************************/
char ToeLift (void)
{
  char toeoff = NONE;
  int ankle;
  ankle = Potentiometer(inback);
  if (inback == IN & ankle >= ToeOffIn)
  {
    toeoff = IN;
  }
  else if (inback == OUT & ankle >= ToeOffOut)
  {
    toeoff = OUT;
  }
  return toeoff;
}

/*******************************************************************************
* FUNCTION NAME: HipSwitch
* PURPOSE:       Determine if hip switch (Sense C) occurs
* CALLED FROM:   State Machine Function
* ARGUMENTS:     none
* RETURNS:       TRUE or FALSE
*******************************************************************************/
char HipSwitch (void)
{
  char hipswitch = FALSE;
  int ana = Potentiometer(HIP);
  if (infrontlast != infront)
  {
    hipswitch = TRUE;
    infrontlast = infront;
  }
  else
  { 
    hipswitch = FALSE;
    infrontlast = infront;
  }
  return hipswitch;
}

/*******************************************************************************
* Action Functions
*******************************************************************************/

/*******************************************************************************
* FUNCTION NAME: AnklePrepare
* PURPOSE:       Prepares Ankle (Action 1)
* CALLED FROM:   State Machine Function
* ARGUMENTS:     none
* RETURNS:       none
*******************************************************************************/
void AnklePrepare(void)
{
  if (infront == IN)
  {
    if (Potentiometer(HIP) > PREPAREIN)
    {
      SendPWM(infront, Position2PWM(PreparedIn, infront, 
              PrepareTolerance, PrepareScale, PrepareOffset));
    }
    else 
    {
      SendPWM(infront, NEUTRAL);
    }
  }
  if (infront == OUT)
  {
    if (Potentiometer(HIP) < PREPAREOUT)
    {
      SendPWM(infront, Position2PWM(PreparedOut, infront, 
              PrepareTolerance, PrepareScale, PrepareOffset));
    }
    else 
    {
      SendPWM(infront, NEUTRAL);
    }
  }
}

/*******************************************************************************
* FUNCTION NAME: AnklePush
* PURPOSE:       Pushes Ankle (Action 2)
* CALLED FROM:   State Machine Function
* ARGUMENTS:     none
* RETURNS:       none
*******************************************************************************/
void AnklePush (void)						
{
  SendPWM(inback,FULL);
}

/*******************************************************************************
* FUNCTION NAME: AnkleRetract
* PURPOSE:       Retracts Ankle (Action 3)
* CALLED FROM:   State Machine Function
* ARGUMENTS:     none
* RETURNS:       none
*******************************************************************************/
void AnkleRetract(void)
{  
  SendPWM(inback, Position2PWM(Retracted, inback, 
          RetractTolerance, RetractScale, RetractOffset));
}			

/*******************************************************************************
* FUNCTION NAME: AnkleLock
* PURPOSE:       Locks support ankle (Action 4)
* CALLED FROM:   State Machine Function
* ARGUMENTS:     none
* RETURNS:       none
*******************************************************************************/
void AnkleLock (void)
{
  if (State == StateII | State == StateIII)
  {
    SendPWM(infront, Position2PWM(PreparedIn, infront, 
            AnkleLockTolerance, AnkleLockScale, AnkleLockOffset));
  }
  if (State == StateIV)
  {
    SendPWM(inback, Position2PWM(PreparedIn, inback, 
            AnkleLockTolerance, AnkleLockScale, AnkleLockOffset));
  }
}

/*******************************************************************************
* FUNCTION NAME: HipSwing
* PURPOSE:       Swings Hip (Action 5)
* CALLED FROM:   State Machine Function
* ARGUMENTS:     none
* RETURNS:       none
*******************************************************************************/
void HipSwing (void)
{
  if (State == StateIII & Potentiometer(inback) < 300)
  {
    if (inback == IN)
    {
      SendPWM(HIP, Position2PWM(SwingIn, HIP, HipSwingTolerance, 
              HipSwingScaleInIII, HipSwingOffsetInIII));
    }
    else
    {
      SendPWM(HIP, Position2PWM(SwingOut, HIP, HipSwingTolerance, 
              HipSwingScaleOutIII, HipSwingOffsetOutIII));
    }
  }
  if (State == StateIV)
  {
    if (infront == IN)
    {
      SendPWM(HIP, Position2PWM(SwingIn, HIP, HipSwingTolerance, 
              HipSwingScaleInIV, HipSwingOffsetInIV));

    }
    else
    {
      SendPWM(HIP, Position2PWM(SwingOut, HIP, HipSwingTolerance, 
              HipSwingScaleOutIV, HipSwingOffsetOutIV));
    }
  }
}

/*******************************************************************************
* State Machine Function
*******************************************************************************/

/*******************************************************************************
* FUNCTION NAME: Process_Data_From_Local_IO
* PURPOSE:       Execute user's realtime code.
* You should modify this routine by adding code which you wish to run fast.
* It will be executed every program loop, and not wait for fresh data 
* from the master processor.
* CALLED FROM:   main.c
* ARGUMENTS:     none
* RETURNS:       void
*******************************************************************************/
void Process_Data_From_Local_IO(void)
{
  // Begin Initial State

  InFrontInBack();						// Determine which pair of legs is in front

  switch (State)
  {
    case StateI:						// If in State I
    {
      AnklePrepare();					// Prepare Front Ankle (Action 1)
      SendPWM(inback, NEUTRAL);			// Send no power to rear ankle
      SendPWM(HIP, NEUTRAL);			// Send no power to hip
      if (HeelStrike() == infront)		// If Heel Strike (Sense A)
      { 
        State = StateII; 				// Assume State II
      }
      break;
    }
    // Begin Walking Cycle

    case StateII:							// If in State II
    {
      SendPWM(HIP, NEUTRAL);			// Send no power to hip
      AnklePush();      				// Push Rear Ankle (Action 2)
      AnkleLock();						// Lock Front Ankle (Action 4)
      if (ToeLift())					// If Toe Lift (Sense B)
      {
        State = StateIII;				// Assume State III
      }
      break;
    }

    case StateIII:						// If in State III
    {
      AnkleRetract();					// Retract Rear Ankle (Action 3)
      AnkleLock();						// Lock Front Ankle (Action 4)
      HipSwing();						// Swing Hip (Action 5)
      if (HipSwitch()) 					// If Hip Switch (Sense C)
      {
        State = StateIV;				// Assume State IV
      }
      break;
    }

    case StateIV:						// If in State IV
    {
      AnkleLock();						// Lock Rear Ankle (Action 4)
      HipSwing();						// Swing Hip (Action 5)
      AnklePrepare();					// Prepare Front Ankle (Action 1)
      if (HeelStrike())					// If Heel Strike (Sense A)
      {
        State = StateII;				// Assume State II
      }
      break;
    }
  }
  // Send PWM signals set by action functions
  Generate_Pwms(MotorInPWM, MotorOutPWM, MotorHipPWM, 131,131,131,131,131); 
}

//Copyright Matt Haberland, 2005
