/*******************************************************************************
* FILE NAME: user_routines.h
*
* DESCRIPTION: 
*  This is the include file which corresponds to user_routines.c and
*  user_routines_fast.c
*  It contains some aliases and function prototypes used in those files.
*
* USAGE:
*  If you add your own routines to those files, this is a good place to add
*  your custom macros (aliases), type definitions, and function prototypes.
*******************************************************************************/

#ifndef __user_program_h_
#define __user_program_h_

/*******************************************************************************
                            MACRO DECLARATIONS
*******************************************************************************/
/* Add your macros (aliases and constants) here.                              */
/* Do not edit the ones in ifi_aliases.h                                      */
/* Macros are substituted in at compile time and make your code more readable */
/* as well as making it easy to change a constant value in one place, rather  */
/* than at every place it is used in your code.                               */
/*
 EXAMPLE CONSTANT:
#define PI_VAL          3.1415

 EXAMPLE ALIAS:
#define LIMIT_SWITCH_1  rc_dig_int1  (Points to another macro in ifi_aliases.h)
*/

/* Used in limit switch routines in user_routines.c */
#define OPEN        1     /* Limit switch is open (input is floating high). */
#define CLOSED      0     /* Limit switch is closed (input connected to ground). */

// PWM Constants
#define FULL					255				// Full PWM Signal
#define NEUTRAL					0				// Neutral PWM Signal

/* Motor Constants - Assign constants for referencing motors and corresponding
   potentiometers in code */
#define NONE					0				// No Motor
#define IN						1				// Inner Motor
#define OUT						2				// Outer Motor
#define HIP						3				// Hip motor

// Logical Constants
#define TRUE					1
#define FALSE					0

// States
#define StateI					0				// Numeral N corresponds to int. n = |N|-1
#define StateII					1
#define StateIII				2
#define StateIV					3

// Positions
#define Retracted				40				// Ankle Pot. reading, retracted position
#define PreparedIn				370				// Inner Pot. reading, prepared position
#define PreparedOut				340				// Outer Pot. reading, prepared position
#define ToeOffIn				PreparedIn+70	// Inner Pot. reading, Toe Off position
#define ToeOffOut				PreparedOut+15	// Outer Pot. reading, Prepared position
#define HIPSWITCH 				315				// Hip Pot. reading, Hip Switch
#define SwingIn					HIPSWITCH+105	// Hip Pot. reading, inner step
#define SwingOut				HIPSWITCH-105	// Hip Pot. reading, outer step
#define PREPAREIN				SwingIn-30		// Hip Pot. reading, inner Ankle Prepare 
#define PREPAREOUT				SwingOut+30		// Hip Pot. reading, outer Ankle Prepare

// Tolerances
#define PrepareTolerance  	 	3
#define RetractTolerance    	3
#define AnkleLockTolerance  	3
#define HipSwingTolerance   	5				

//  Scaling Factors
#define HipSwingScaleInIII		30				// Inner leg swing, State II
#define HipSwingScaleInIV   	20				// Inner leg swing, State III
#define HipSwingScaleOutIII		40				// Outer leg swing, State II
#define HipSwingScaleOutIV   	20				// Outer leg swing, State III
#define AnkleLockScale 			9
#define PrepareScale   			8
#define RetractScale    		8

// Offsets
#define HipSwingOffsetInIII		30				// Inner leg swing, State II
#define HipSwingOffsetInIV		20				// Inner leg swing, State III
#define HipSwingOffsetOutIII	40				// Outer leg swing, State II
#define HipSwingOffsetOutIV		20				// Outer leg swing, State III
#define AnkleLockOffset 		9
#define PrepareOffset   		7
#define RetractOffset    		7

// Input Referencing Macros
#define PotIn					rc_ana_in01		// Inner Potentiometer
#define PotOut					rc_ana_in02		// Outer Potentiometer
#define PotHip					rc_ana_in03		// Hip Potentiometer

#define LimitHeelIn 			rc_dig_int1		// Inner Heel Limit Switch
#define LimitHeelOut 			rc_dig_int4		// Outer Heel Limit Switch

#define MotorInPWM				pwm01			// Inner Motor
#define MotorOutPWM				pwm02			// Outer Motor
#define MotorHipPWM				pwm03			// Hip Motor
 /*******************************************************************************
                           FUNCTION PROTOTYPES
*******************************************************************************/

/* These routines reside in user_routines.c */
void User_Initialization(void);
void Process_Data_From_Master_uP(void);

/* These routines reside in user_routines_fast.c */
void InterruptHandlerLow (void);  /* DO NOT CHANGE! */

// Processor Functions
int Potentiometer (char motor);	// Continuous Feedback Alpha and Beta
void SendPWM(char motor, int pwm);
void InFrontInBack(void);
int Position2PWM(int Ptarget, char motor, char tolerance, char scale, char offset);

// Sense Functions
char HeelStrike(void);				// Sense A
char ToeLift(void);					// Sense B
char HipSwitch(void);				// Sense C

// Action Functions
void AnklePrepare(void);			// Action 1
void AnklePush(void);				// Action 2
void AnkleRetract(void);			// Action 3
void AnkleLock (void);				// Action 4
void HipSwing(void);				// Action 5

// State Machine Function
void Process_Data_From_Local_IO(void);	// States I-IV

#endif

//Copyright Matt Haberland, 2005
